/** @file   filelist.cpp
 * @brief   Implementation of FileList - class.
 * @version $Revision: 1.2 $
 * @author  Tomi Lamminsaari
 */

#include "filelist.h"
#include <fstream>
#include "datatypes.h"
#include "eng2dPrivateConstants.h"

using std::string;
using std::vector;
using std::ifstream;
using std::istream;
using std::ofstream;
using std::endl;

namespace eng2d {

///
/// Static members, constants and datatypes
/// =======================================

#define FILELIST_OPENINGTAG "<eng2d_filelist>"
#define FILELIST_CLOSINGTAG "</eng2d_filelist>"
#define FILELIST_FILEPARAMNAME  "file:"

///
/// Constructors, destructor and operators
/// ======================================

/** Constructor
 */
FileList::FileList() :
  m_bad( false )
{
}



/** Constructor.
 */
FileList::FileList( const string& filename ) throw (xIoError) :
  m_bad( false )
{
  if ( this->load( filename ) != KErrNone ) {
    throw xIoError( "FileList",
                    "FileList(const string&)",
                    "Failed to load Animation" );
  }
}



/** Destructor
 */
FileList::~FileList()
{
}



/** Copy constructor.
 */
FileList::FileList( const FileList& rFL ) :
  m_bad( rFL.m_bad ),
  m_filelist( rFL.m_filelist )
{
}



/** Assignment operator
 */
FileList& FileList::operator= ( const FileList& rFL )
{
  if ( this != &rFL ) {
    m_bad = rFL.m_bad;
    m_filelist = rFL.m_filelist;
  }
  return *this;
}




///
/// Public methods
/// ==============

/** Adds new filename to this filelist.
 */
void FileList::addFile( const string& filename )
{
  m_filelist.push_back( filename );
}



/** Clears this filelist.
 */
void FileList::clear()
{
  m_bad = false;
  m_filelist.clear();
}



/** Deletes the index'th filename
 */
void FileList::delFile( int index )
{
  if ( this->validIndex( index ) == false ) {
    return;
  }
  vector<string>::iterator it = m_filelist.begin() + index;
  m_filelist.erase( it );
}



/** Sets the index'th filename.
 */
void FileList::setFile( int index, const string& filename )
{
  if ( this->validIndex( index ) == false ) {
    return;
  }
  m_filelist.at( index ) = filename;
}



/** Deletes the find occurance of given filename
 */
void FileList::delFile( const string& filename )
{
  int index = this->contains( filename );
  if ( index != -1 ) {
    this->delFile( index );
  }
}



/** Loads the filelist.
 */
int FileList::load( const string& filename )
{
  this->clear();
  m_bad = true;
  
  ifstream fin( filename.c_str() );
  if ( !fin ) {
    return KErrNotFound;
  }
  if ( this->readHeader( fin ) == true ) {
    fin.close();
    return KErrNotSupported;
  }
  
  while ( true ) {
    if ( fin.eof() == true ) {
      fin.close();
      return KErrEof;
    }
    
    string tmp;
    fin >> tmp;
    if ( tmp == FILELIST_FILEPARAMNAME ) {
      string fname;
      fin >> fname;
      this->addFile( fname );
      
    } else if ( tmp == FILELIST_CLOSINGTAG ) {
      break;
      
    } else if ( tmp == KStrCommentLine ) {
      fin.ignore( KMaxLineLength, KCharNewLine );
      
    } else {
      return KErrCorrupt;
      
    }
  }
  fin.close();
  m_bad = false;
  return KErrNone;
}



/** Saves the filelist
 */
int FileList::save( const string& filename ) const
{
  ofstream fout( filename.c_str() );
  if ( !fout ) {
    return KErrNotFound;
  }
  
  string indent = "  ";
  fout << FILELIST_OPENINGTAG << endl;
  
  for ( int i=0; i < this->filecount(); i++ ) {
    fout << indent << FILELIST_FILEPARAMNAME << " " << this->getFile( i ) << endl;
  }
  fout << FILELIST_CLOSINGTAG << endl;
  fout.close();
  
  return KErrNone;
}




///
/// Getter methods
/// ==============

/** Tells if this filelist is not ok.
 */
bool FileList::bad() const
{
  return m_bad;
}



/** Returns the index'th filename
 */
string FileList::getFile( int index ) const
{
  if ( this->validIndex( index ) == false ) {
    return string("");
  }
  return m_filelist.at( index );
}



/** Returns the number of files this filelist has.
 */
int FileList::filecount() const
{
  return m_filelist.size();
}



/** Finds the index of given filename
 */
int FileList::contains( const string& searchFor ) const
{
  for ( int i=0; i < this->filecount(); i++ ) {
    string tmp = this->getFile( i );
    if ( tmp == searchFor ) {
      return i;
    }
  }
  return KErrNotFound;
}




///
/// Private or Protected methods
/// ============================

/** Tells if given index is valid index.
 */
bool FileList::validIndex( int index ) const
{
  if ( index < 0 || index >= this->filecount() ) {
    return false;
  }
  return true;
}



/** Finds the opening tag from given stream
 */
bool FileList::readHeader( istream& rIn ) const
{
  while ( true ) {
    if ( rIn.eof() == true ) {
      return true;
    }
    
    string tmp;
    rIn >> tmp;
    if ( tmp == FILELIST_OPENINGTAG ) {
      return false;
    }
  }
}


} // end of namespace
